<?php
// Admin Panel for Power Streams - Enhanced Version

// Password Configuration
$admin_password = "admin123"; // Change this to your desired password
$session_timeout = 1800; // 30 minutes in seconds

// Set default timezone to London
date_default_timezone_set('Europe/London');

// Start session and check authentication
session_start();

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    if ($_POST['password'] === $admin_password) {
        $_SESSION['admin_authenticated'] = true;
        $_SESSION['admin_login_time'] = time();
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        $login_error = "Invalid password!";
    }
}

// Check if user is authenticated
$is_authenticated = isset($_SESSION['admin_authenticated']) && 
                   $_SESSION['admin_authenticated'] === true &&
                   (time() - $_SESSION['admin_login_time']) < $session_timeout;

// Update login time on each request
if ($is_authenticated) {
    $_SESSION['admin_login_time'] = time();
}

// Logout functionality
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// If not authenticated, show login page
if (!$is_authenticated) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Login - Power Streams</title>
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            :root {
                --primary-color: #0f172a;
                --secondary-color: #1e293b;
                --accent-color: #1E40AF;
                --accent-red: #DC2626;
                --text-color: #e2e8f0;
            }
            
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
                font-family: 'Inter', system-ui, sans-serif;
            }
            
            body {
                background: var(--primary-color);
                color: var(--text-color);
                display: flex;
                justify-content: center;
                align-items: center;
                min-height: 100vh;
                padding: 20px;
                background-image: linear-gradient(135deg, var(--primary-color) 0%, #1e293b 100%);
            }
            
            .login-container {
                background: var(--secondary-color);
                padding: 40px;
                border-radius: 15px;
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
                width: 100%;
                max-width: 450px;
                border: 1px solid #334155;
                position: relative;
                overflow: hidden;
            }
            
            .login-container::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 4px;
                background: linear-gradient(90deg, var(--accent-color), var(--accent-red));
            }
            
            .logo {
                text-align: center;
                margin-bottom: 30px;
            }
            
            .logo-text {
                font-size: 2.5rem;
                font-weight: 800;
                color: white;
                margin-bottom: 5px;
            }
            
            .power {
                color: var(--accent-red);
            }
            
            .admin-title {
                color: #94a3b8;
                font-size: 1.1rem;
                font-weight: 500;
            }
            
            .login-form {
                margin-top: 20px;
            }
            
            .form-group {
                margin-bottom: 25px;
            }
            
            label {
                display: block;
                margin-bottom: 10px;
                font-weight: 600;
                color: var(--text-color);
                font-size: 14px;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            
            .input-with-icon {
                position: relative;
            }
            
            .input-icon {
                position: absolute;
                left: 15px;
                top: 50%;
                transform: translateY(-50%);
                color: #64748b;
            }
            
            input {
                width: 100%;
                padding: 15px 15px 15px 45px;
                border: 2px solid #334155;
                border-radius: 8px;
                background: var(--primary-color);
                color: var(--text-color);
                font-size: 16px;
                transition: all 0.3s ease;
            }
            
            input:focus {
                outline: none;
                border-color: var(--accent-color);
                box-shadow: 0 0 0 3px rgba(30, 64, 175, 0.2);
            }
            
            button {
                width: 100%;
                padding: 15px;
                background: linear-gradient(135deg, var(--accent-color), #1E3A8A);
                color: white;
                border: none;
                border-radius: 8px;
                font-weight: 600;
                font-size: 16px;
                cursor: pointer;
                transition: all 0.3s ease;
                position: relative;
                overflow: hidden;
            }
            
            button:hover {
                transform: translateY(-2px);
                box-shadow: 0 10px 20px rgba(30, 64, 175, 0.3);
            }
            
            button:active {
                transform: translateY(0);
            }
            
            .error-message {
                background: rgba(220, 38, 38, 0.1);
                color: #fca5a5;
                padding: 12px;
                border-radius: 8px;
                margin-bottom: 20px;
                text-align: center;
                border: 1px solid rgba(220, 38, 38, 0.3);
            }
            
            .login-info {
                margin-top: 25px;
                padding: 15px;
                background: rgba(30, 64, 175, 0.1);
                border-radius: 8px;
                border-left: 4px solid var(--accent-color);
                font-size: 14px;
                color: #94a3b8;
            }
            
            .time-info {
                text-align: center;
                margin-top: 15px;
                font-size: 12px;
                color: #64748b;
            }
        </style>
    </head>
    <body>
        <div class="login-container">
            <div class="logo">
                <div class="logo-text"><span class="power">Power</span> Streams</div>
                <div class="admin-title">Admin Panel Login</div>
            </div>
            
            <?php if (isset($login_error)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-triangle"></i> <?php echo $login_error; ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" class="login-form">
                <div class="form-group">
                    <label for="password">Admin Password</label>
                    <div class="input-with-icon">
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" id="password" name="password" required 
                               placeholder="Enter your admin password">
                    </div>
                </div>
                
                <button type="submit" name="login">
                    <i class="fas fa-sign-in-alt"></i> Login to Admin Panel
                </button>
            </form>
            
            <div class="login-info">
                <i class="fas fa-info-circle"></i> 
                Enter the admin password to access the management panel.
            </div>
            
            <div class="time-info">
                <i class="fas fa-clock"></i> London Time: <?php echo date('H:i:s'); ?>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// If authenticated, continue with the admin panel

// Initialize data
$data_file = 'data/matches.json';
$data = [];

// Load existing data
if (file_exists($data_file)) {
    $json_data = file_get_contents($data_file);
    $data = json_decode($json_data, true) ?: [];
}

// Initialize arrays
$matches = $data['matches'] ?? [];
$categories = $data['categories'] ?? [];

// Function to check if a match has expired based on London time and duration
function isMatchExpired($match) {
    if (!isset($match['date']) || !isset($match['time']) || !isset($match['duration'])) {
        return true;
    }
    
    // Create London timezone
    $londonTz = new DateTimeZone('Europe/London');
    
    // Create match start time in London timezone
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return true;
    }
    
    // Parse duration (HH:MM format)
    $durationParts = explode(':', $match['duration']);
    $hours = intval($durationParts[0]);
    $minutes = intval($durationParts[1] ?? 0);
    
    // Calculate end time by adding duration to start time
    $endTime = clone $startTime;
    $endTime->modify("+{$hours} hours +{$minutes} minutes");
    
    // Get current time in London
    $currentTime = new DateTime('now', $londonTz);
    
    // Check if current time is after end time
    return $currentTime > $endTime;
}

// Function to check if stream link should be clickable (30 minutes before match)
function isStreamClickable($match) {
    if (!isset($match['date']) || !isset($match['time'])) {
        return false;
    }
    
    // Create London timezone
    $londonTz = new DateTimeZone('Europe/London');
    
    // Create match start time in London timezone
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return false;
    }
    
    // Calculate 30 minutes before start time
    $clickableTime = clone $startTime;
    $clickableTime->modify("-30 minutes");
    
    // Get current time in London
    $currentTime = new DateTime('now', $londonTz);
    
    // Check if current time is within 30 minutes before start time or after
    return $currentTime >= $clickableTime;
}

// Function to get match status with London time
function getMatchStatus($match) {
    if (isMatchExpired($match)) {
        return 'expired';
    }
    
    $londonTz = new DateTimeZone('Europe/London');
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    $currentTime = new DateTime('now', $londonTz);
    
    if ($currentTime >= $startTime) {
        return 'live';
    } else {
        return 'upcoming';
    }
}

// Function to get category order
function getCategoryOrder($categoryName, $categories) {
    foreach ($categories as $category) {
        if ($category['name'] === $categoryName) {
            return $category['order'];
        }
    }
    return 999; // High number for categories not found
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Add Category
    if (isset($_POST['add_category'])) {
        $new_category = trim($_POST['category_name']);
        if ($new_category && !in_array($new_category, array_column($categories, 'name'))) {
            $categories[] = [
                'name' => $new_category, 
                'order' => count($categories),
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
    }
    
    // Delete Category
    if (isset($_POST['delete_category'])) {
        $category_to_delete = $_POST['category_name'];
        $categories = array_filter($categories, function($cat) use ($category_to_delete) {
            return $cat['name'] !== $category_to_delete;
        });
        // Reindex categories with proper order
        $categories = array_values($categories);
        foreach ($categories as $index => &$cat) {
            $cat['order'] = $index;
        }
    }
    
    // Move Category Up
    if (isset($_POST['move_category_up'])) {
        $category_name = $_POST['category_name'];
        $index = null;
        
        // Find the index of the category to move
        foreach ($categories as $i => $cat) {
            if ($cat['name'] === $category_name) {
                $index = $i;
                break;
            }
        }
        
        if ($index !== null && $index > 0) {
            // Swap the categories
            $temp = $categories[$index];
            $categories[$index] = $categories[$index - 1];
            $categories[$index - 1] = $temp;
            
            // Update order values
            $categories[$index]['order'] = $index;
            $categories[$index - 1]['order'] = $index - 1;
        }
    }
    
    // Move Category Down
    if (isset($_POST['move_category_down'])) {
        $category_name = $_POST['category_name'];
        $index = null;
        
        // Find the index of the category to move
        foreach ($categories as $i => $cat) {
            if ($cat['name'] === $category_name) {
                $index = $i;
                break;
            }
        }
        
        if ($index !== null && $index < count($categories) - 1) {
            // Swap the categories
            $temp = $categories[$index];
            $categories[$index] = $categories[$index + 1];
            $categories[$index + 1] = $temp;
            
            // Update order values
            $categories[$index]['order'] = $index;
            $categories[$index + 1]['order'] = $index + 1;
        }
    }
    
    // Add Match
    if (isset($_POST['add_match'])) {
        $new_match = [
            'id' => uniqid(),
            'category' => $_POST['match_category'],
            'date' => $_POST['match_date'],
            'time' => $_POST['match_time'],
            'duration' => $_POST['match_duration'],
            'league' => $_POST['match_league'],
            'team1' => $_POST['match_team1'],
            'team2' => $_POST['match_team2'],
            'stream_url' => $_POST['match_stream_url'],
            'order' => count($matches),
            'created_at' => date('Y-m-d H:i:s')
        ];
        $matches[] = $new_match;
    }
    
    // Edit Match
    if (isset($_POST['edit_match'])) {
        $match_id = $_POST['match_id'];
        foreach ($matches as &$match) {
            if ($match['id'] === $match_id) {
                $match['category'] = $_POST['match_category'];
                $match['date'] = $_POST['match_date'];
                $match['time'] = $_POST['match_time'];
                $match['duration'] = $_POST['match_duration'];
                $match['league'] = $_POST['match_league'];
                $match['team1'] = $_POST['match_team1'];
                $match['team2'] = $_POST['match_team2'];
                $match['stream_url'] = $_POST['match_stream_url'];
                $match['updated_at'] = date('Y-m-d H:i:s');
                break;
            }
        }
    }
    
    // Delete Match
    if (isset($_POST['delete_match'])) {
        $match_id = $_POST['match_id'];
        $matches = array_filter($matches, function($match) use ($match_id) {
            return $match['id'] !== $match_id;
        });
        $matches = array_values($matches);
        // Reindex matches with proper order
        foreach ($matches as $index => &$match) {
            $match['order'] = $index;
        }
    }
    
    // Move Match Up
    if (isset($_POST['move_match_up'])) {
        $match_id = $_POST['match_id'];
        $index = null;
        
        // Find the index of the match to move
        foreach ($matches as $i => $match) {
            if ($match['id'] === $match_id) {
                $index = $i;
                break;
            }
        }
        
        if ($index !== null && $index > 0) {
            // Swap the matches
            $temp = $matches[$index];
            $matches[$index] = $matches[$index - 1];
            $matches[$index - 1] = $temp;
            
            // Update order values
            $matches[$index]['order'] = $index;
            $matches[$index - 1]['order'] = $index - 1;
        }
    }
    
    // Move Match Down
    if (isset($_POST['move_match_down'])) {
        $match_id = $_POST['match_id'];
        $index = null;
        
        // Find the index of the match to move
        foreach ($matches as $i => $match) {
            if ($match['id'] === $match_id) {
                $index = $i;
                break;
            }
        }
        
        if ($index !== null && $index < count($matches) - 1) {
            // Swap the matches
            $temp = $matches[$index];
            $matches[$index] = $matches[$index + 1];
            $matches[$index + 1] = $temp;
            
            // Update order values
            $matches[$index]['order'] = $index;
            $matches[$index + 1]['order'] = $index + 1;
        }
    }
    
    // Clear All Matches
    if (isset($_POST['clear_all_matches'])) {
        $matches = [];
    }
    
    // Clear All Categories
    if (isset($_POST['clear_all_categories'])) {
        $categories = [];
    }
    
    // Remove expired matches before saving
    $matches = array_filter($matches, function($match) {
        return !isMatchExpired($match);
    });
    
    // Sort categories by order
    usort($categories, function($a, $b) {
        return $a['order'] - $b['order'];
    });
    
    // Sort matches by category order first, then by match order
    usort($matches, function($a, $b) use ($categories) {
        $aCategoryOrder = getCategoryOrder($a['category'], $categories);
        $bCategoryOrder = getCategoryOrder($b['category'], $categories);
        
        if ($aCategoryOrder !== $bCategoryOrder) {
            return $aCategoryOrder - $bCategoryOrder;
        }
        
        return $a['order'] - $b['order'];
    });
    
    // Save data to JSON file
    $data = [
        'matches' => $matches,
        'categories' => $categories,
        'last_updated' => date('Y-m-d H:i:s')
    ];
    
    // Create data directory if it doesn't exist
    if (!is_dir('data')) {
        mkdir('data', 0755, true);
    }
    
    file_put_contents($data_file, json_encode($data, JSON_PRETTY_PRINT));
    
    // Redirect to avoid form resubmission
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Get match for editing
$edit_match = null;
if (isset($_GET['edit_match'])) {
    $match_id = $_GET['edit_match'];
    foreach ($matches as $match) {
        if ($match['id'] === $match_id) {
            $edit_match = $match;
            break;
        }
    }
}

// Remove expired matches on page load (in case they weren't removed during POST)
$matches = array_filter($matches, function($match) {
    return !isMatchExpired($match);
});

// Sort categories by order for display
usort($categories, function($a, $b) {
    return $a['order'] - $b['order'];
});

// Sort matches by category order first, then by match order for display
usort($matches, function($a, $b) use ($categories) {
    $aCategoryOrder = getCategoryOrder($a['category'], $categories);
    $bCategoryOrder = getCategoryOrder($b['category'], $categories);
    
    if ($aCategoryOrder !== $bCategoryOrder) {
        return $aCategoryOrder - $bCategoryOrder;
    }
    
    return $a['order'] - $b['order'];
});

// Save the updated matches list if any were removed
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $data = [
        'matches' => $matches,
        'categories' => $categories,
        'last_updated' => date('Y-m-d H:i:s')
    ];
    
    if (!is_dir('data')) {
        mkdir('data', 0755, true);
    }
    
    file_put_contents($data_file, json_encode($data, JSON_PRETTY_PRINT));
}

// Calculate statistics
$total_matches = count($matches);
$live_matches = count(array_filter($matches, function($match) { return getMatchStatus($match) === 'live'; }));
$upcoming_matches = count(array_filter($matches, function($match) { return getMatchStatus($match) === 'upcoming'; }));
$clickable_streams = count(array_filter($matches, 'isStreamClickable'));
$total_categories = count($categories);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Power Streams</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #0f172a;
            --secondary-color: #1e293b;
            --accent-color: #1E40AF;
            --accent-red: #DC2626;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --live-color: #ef4444;
            --upcoming-color: #3b82f6;
            --text-color: #e2e8f0;
            --border-color: #334155;
            --sidebar-width: 280px;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Inter', system-ui, sans-serif;
        }
        
        body {
            background-color: var(--primary-color);
            color: var(--text-color);
            line-height: 1.6;
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar */
        .sidebar {
            width: var(--sidebar-width);
            background: var(--secondary-color);
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            overflow-y: auto;
            border-right: 1px solid var(--border-color);
            display: flex;
            flex-direction: column;
        }
        
        .logo {
            padding: 25px 20px;
            text-align: center;
            border-bottom: 1px solid var(--border-color);
            background: linear-gradient(135deg, var(--secondary-color), #1e293b);
        }
        
        .logo-text {
            font-size: 1.8rem;
            font-weight: 800;
            color: white;
            margin-bottom: 5px;
        }
        
        .power {
            color: var(--accent-red);
        }
        
        .admin-title {
            font-size: 0.9rem;
            color: #94a3b8;
            font-weight: 500;
        }
        
        .time-info {
            font-size: 0.8rem;
            color: #64748b;
            margin-top: 8px;
        }
        
        .nav-links {
            padding: 20px 0;
            flex: 1;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: var(--text-color);
            text-decoration: none;
            transition: all 0.3s ease;
            border-left: 4px solid transparent;
            margin: 5px 10px;
            border-radius: 8px;
        }
        
        .nav-link:hover {
            background: rgba(255, 255, 255, 0.05);
            border-left-color: var(--accent-color);
            transform: translateX(5px);
        }
        
        .nav-link.active {
            background: rgba(30, 64, 175, 0.2);
            border-left-color: var(--accent-color);
            color: var(--accent-color);
            font-weight: 600;
        }
        
        .nav-link i {
            margin-right: 12px;
            width: 20px;
            text-align: center;
            font-size: 1.1rem;
        }
        
        .user-info {
            padding: 20px;
            border-top: 1px solid var(--border-color);
            background: rgba(255, 255, 255, 0.02);
        }
        
        .user-details {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            background: var(--accent-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }
        
        .user-name {
            font-weight: 600;
            color: var(--text-color);
        }
        
        .user-role {
            font-size: 0.8rem;
            color: #94a3b8;
        }
        
        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: var(--sidebar-width);
            padding: 0;
            min-height: 100vh;
        }
        
        .header {
            background: var(--secondary-color);
            padding: 20px 30px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .page-title {
            font-size: 1.8rem;
            color: var(--text-color);
            font-weight: 700;
        }
        
        .header-actions {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
        }
        
        .btn-primary {
            background: var(--accent-color);
            color: white;
        }
        
        .btn-primary:hover {
            background: #1E3A8A;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(30, 64, 175, 0.3);
        }
        
        .btn-danger {
            background: var(--accent-red);
            color: white;
        }
        
        .btn-danger:hover {
            background: #b91c1c;
            transform: translateY(-2px);
        }
        
        .btn-success {
            background: var(--success-color);
            color: white;
        }
        
        .btn-success:hover {
            background: #0d9660;
            transform: translateY(-2px);
        }
        
        .btn-warning {
            background: var(--warning-color);
            color: white;
        }
        
        .btn-warning:hover {
            background: #d97706;
            transform: translateY(-2px);
        }
        
        .content-area {
            padding: 30px;
        }
        
        /* Statistics Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: var(--secondary-color);
            padding: 25px;
            border-radius: 12px;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: var(--accent-color);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .stat-card.live::before {
            background: var(--live-color);
        }
        
        .stat-card.upcoming::before {
            background: var(--upcoming-color);
        }
        
        .stat-card.success::before {
            background: var(--success-color);
        }
        
        .stat-icon {
            font-size: 2.5rem;
            margin-bottom: 15px;
            opacity: 0.8;
        }
        
        .stat-number {
            font-size: 2.2rem;
            font-weight: 800;
            margin-bottom: 5px;
        }
        
        .stat-label {
            font-size: 14px;
            color: #94a3b8;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            font-weight: 600;
        }
        
        /* Sections */
        .section {
            background: var(--secondary-color);
            border-radius: 12px;
            padding: 30px;
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }
        
        .section-title {
            font-size: 1.4rem;
            margin-bottom: 25px;
            color: var(--text-color);
            font-weight: 700;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .section-actions {
            display: flex;
            gap: 10px;
        }
        
        /* Forms */
        .form-grid {
            display: grid;
            gap: 20px;
            margin-bottom: 25px;
        }
        
        .grid-2 {
            grid-template-columns: 1fr 1fr;
        }
        
        .grid-3 {
            grid-template-columns: 1fr 1fr 1fr;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--text-color);
            font-size: 14px;
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            background: var(--primary-color);
            color: var(--text-color);
            font-size: 14px;
            transition: all 0.3s ease;
        }
        
        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(30, 64, 175, 0.2);
        }
        
        /* Tables */
        .table-container {
            background: var(--primary-color);
            border-radius: 8px;
            overflow: hidden;
            border: 1px solid var(--border-color);
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
            background: var(--primary-color);
        }
        
        .table th {
            background: var(--accent-color);
            padding: 15px 20px;
            text-align: left;
            font-weight: 600;
            font-size: 14px;
            color: white;
            border-bottom: 2px solid var(--accent-color);
        }
        
        .table td {
            padding: 15px 20px;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }
        
        .table tr:last-child td {
            border-bottom: none;
        }
        
        .table tr:hover {
            background: rgba(255, 255, 255, 0.03);
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            border-radius: 6px;
        }
        
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-live {
            background: var(--live-color);
            color: white;
        }
        
        .status-upcoming {
            background: var(--upcoming-color);
            color: white;
        }
        
        .status-expired {
            background: #6b7280;
            color: white;
        }
        
        .stream-available {
            color: var(--success-color);
            font-weight: 600;
        }
        
        .stream-pending {
            color: var(--warning-color);
            font-weight: 600;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .sidebar {
                width: 250px;
            }
            
            .main-content {
                margin-left: 250px;
            }
            
            .grid-2, .grid-3 {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            body {
                flex-direction: column;
            }
            
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            
            .main-content {
                margin-left: 0;
            }
            
            .nav-links {
                display: flex;
                overflow-x: auto;
                padding: 15px 10px;
            }
            
            .nav-link {
                flex-shrink: 0;
                border-left: none;
                border-bottom: 3px solid transparent;
                margin: 0 5px;
            }
            
            .nav-link.active {
                border-left: none;
                border-bottom-color: var(--accent-color);
            }
            
            .header {
                padding: 15px 20px;
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .header-actions {
                width: 100%;
                justify-content: space-between;
            }
            
            .content-area {
                padding: 20px;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }
        
        @media (max-width: 480px) {
            .section {
                padding: 20px;
            }
            
            .table {
                display: block;
                overflow-x: auto;
            }
            
            .btn {
                padding: 8px 15px;
                font-size: 13px;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="logo">
            <div class="logo-text"><span class="power">Power</span> Streams</div>
            <div class="admin-title">Admin Control Panel</div>
            <div class="time-info">
                <i class="fas fa-clock"></i> London: <?php echo date('H:i:s'); ?>
            </div>
        </div>
        
        <div class="nav-links">
            <a href="#dashboard" class="nav-link active">
                <i class="fas fa-tachometer-alt"></i> Dashboard
            </a>
            <a href="#matches" class="nav-link">
                <i class="fas fa-tv"></i> Manage Matches
            </a>
            <a href="#categories" class="nav-link">
                <i class="fas fa-list"></i> Categories
            </a>
            <a href="index.php" class="nav-link">
                <i class="fas fa-eye"></i> View Website
            </a>
        </div>
        
        <div class="user-info">
            <div class="user-details">
                <div class="user-avatar">
                    <i class="fas fa-user-shield"></i>
                </div>
                <div>
                    <div class="user-name">Administrator</div>
                    <div class="user-role">Super Admin</div>
                </div>
            </div>
            <a href="?logout" class="btn btn-danger" style="width: 100%; justify-content: center;">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <div class="header">
            <h1 class="page-title">Admin Dashboard</h1>
            <div class="header-actions">
                <div>
                    <span style="color: #94a3b8; font-size: 14px;">
                        Last updated: <?php echo date('M j, Y H:i:s'); ?>
                    </span>
                </div>
                <div>
                    <a href="index.php" class="btn btn-primary">
                        <i class="fas fa-external-link-alt"></i> Visit Site
                    </a>
                </div>
            </div>
        </div>
        
        <div class="content-area">
            <!-- Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-tv"></i>
                    </div>
                    <div class="stat-number"><?php echo $total_matches; ?></div>
                    <div class="stat-label">Total Matches</div>
                </div>
                
                <div class="stat-card live">
                    <div class="stat-icon">
                        <i class="fas fa-broadcast-tower"></i>
                    </div>
                    <div class="stat-number"><?php echo $live_matches; ?></div>
                    <div class="stat-label">Live Matches</div>
                </div>
                
                <div class="stat-card upcoming">
                    <div class="stat-icon">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-number"><?php echo $upcoming_matches; ?></div>
                    <div class="stat-label">Upcoming Matches</div>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-icon">
                        <i class="fas fa-play-circle"></i>
                    </div>
                    <div class="stat-number"><?php echo $clickable_streams; ?></div>
                    <div class="stat-label">Active Streams</div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="section">
                <h2 class="section-title">Quick Actions</h2>
                <div class="action-buttons" style="justify-content: center; gap: 15px;">
                    <a href="#add-match" class="btn btn-success">
                        <i class="fas fa-plus"></i> Add New Match
                    </a>
                    <a href="#add-category" class="btn btn-primary">
                        <i class="fas fa-folder-plus"></i> Add Category
                    </a>
                    <button type="button" class="btn btn-warning" onclick="document.getElementById('clear-matches-form').submit();">
                        <i class="fas fa-trash"></i> Clear All Matches
                    </button>
                </div>
            </div>

            <!-- Add/Edit Match Form -->
            <div class="section" id="add-match">
                <h2 class="section-title">
                    <?php echo $edit_match ? 'Edit Match' : 'Add New Match'; ?>
                </h2>
                
                <form method="POST">
                    <?php if ($edit_match): ?>
                        <input type="hidden" name="match_id" value="<?php echo $edit_match['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-grid grid-2">
                        <div class="form-group">
                            <label for="match_category">Category:</label>
                            <select id="match_category" name="match_category" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo htmlspecialchars($category['name']); ?>"
                                        <?php echo ($edit_match && $edit_match['category'] === $category['name']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="match_league">League/Tournament:</label>
                            <input type="text" id="match_league" name="match_league" required
                                   value="<?php echo $edit_match['league'] ?? ''; ?>"
                                   placeholder="e.g., Premier League, La Liga">
                        </div>
                    </div>
                    
                    <div class="form-grid grid-3">
                        <div class="form-group">
                            <label for="match_date">Match Date:</label>
                            <input type="date" id="match_date" name="match_date" required
                                   value="<?php echo $edit_match['date'] ?? ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="match_time">Start Time (London):</label>
                            <input type="time" id="match_time" name="match_time" required
                                   value="<?php 
                                       if (isset($edit_match['time'])) {
                                           echo $edit_match['time'];
                                       } else {
                                           $londonTime = new DateTime('now', new DateTimeZone('Europe/London'));
                                           echo $londonTime->format('H:i');
                                       }
                                   ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="match_duration">Duration (HH:MM):</label>
                            <input type="text" id="match_duration" name="match_duration" required
                                   value="<?php echo $edit_match['duration'] ?? '02:00'; ?>"
                                   placeholder="e.g., 02:00 for 2 hours">
                        </div>
                    </div>
                    
                    <div class="form-grid grid-2">
                        <div class="form-group">
                            <label for="match_team1">Team 1 (Home):</label>
                            <input type="text" id="match_team1" name="match_team1" required
                                   value="<?php echo $edit_match['team1'] ?? ''; ?>"
                                   placeholder="Home team name">
                        </div>
                        
                        <div class="form-group">
                            <label for="match_team2">Team 2 (Away):</label>
                            <input type="text" id="match_team2" name="match_team2" 
                                   value="<?php echo $edit_match['team2'] ?? ''; ?>"
                                   placeholder="Away team name">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="match_stream_url">Stream URL:</label>
                        <input type="url" id="match_stream_url" name="match_stream_url" required
                               value="<?php echo $edit_match['stream_url'] ?? ''; ?>"
                               placeholder="https://example.com/stream">
                    </div>
                    
                    <div class="form-group">
                        <?php if ($edit_match): ?>
                            <button type="submit" name="edit_match" class="btn btn-success">
                                <i class="fas fa-save"></i> Update Match
                            </button>
                            <a href="admin.php" class="btn btn-warning">
                                <i class="fas fa-times"></i> Cancel Edit
                            </a>
                        <?php else: ?>
                            <button type="submit" name="add_match" class="btn btn-success">
                                <i class="fas fa-plus"></i> Add Match
                            </button>
                        <?php endif; ?>
                    </div>
                </form>
            </div>

            <!-- Matches Management -->
            <div class="section" id="matches">
                <h2 class="section-title">
                    <span>Manage Matches (<?php echo count($matches); ?>)</span>
                    <div class="section-actions">
                        <button type="button" class="btn btn-warning btn-sm" onclick="document.getElementById('clear-matches-form').submit();">
                            <i class="fas fa-trash"></i> Clear All
                        </button>
                    </div>
                </h2>
                
                <?php if (empty($matches)): ?>
                    <div style="text-align: center; padding: 40px; color: #94a3b8;">
                        <i class="fas fa-tv" style="font-size: 3rem; margin-bottom: 15px;"></i>
                        <h3>No Matches Found</h3>
                        <p>Add your first match using the form above.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Match Details</th>
                                    <th>Category</th>
                                    <th>Date & Time</th>
                                    <th>Status</th>
                                    <th>Stream Access</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($matches as $index => $match): ?>
                                    <?php
                                    $status = getMatchStatus($match);
                                    $is_clickable = isStreamClickable($match);
                                    ?>
                                    <tr>
                                        <td>
                                            <div style="font-weight: 600; margin-bottom: 5px;">
                                                <?php echo htmlspecialchars($match['team1']); ?> vs <?php echo htmlspecialchars($match['team2']); ?>
                                            </div>
                                            <div style="font-size: 13px; color: #94a3b8;">
                                                <?php echo htmlspecialchars($match['league']); ?>
                                            </div>
                                        </td>
                                        <td><?php echo htmlspecialchars($match['category']); ?></td>
                                        <td>
                                            <div><?php echo date('M j, Y', strtotime($match['date'])); ?></div>
                                            <div style="font-size: 13px; color: #94a3b8;">
                                                <?php echo $match['time']; ?> London
                                            </div>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo $status; ?>">
                                                <?php echo ucfirst($status); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($is_clickable): ?>
                                                <span class="stream-available">
                                                    <i class="fas fa-check-circle"></i> Available
                                                </span>
                                            <?php else: ?>
                                                <span class="stream-pending">
                                                    <i class="fas fa-clock"></i> Pending
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="action-buttons">
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="match_id" value="<?php echo $match['id']; ?>">
                                                    <button type="submit" name="move_match_up" class="btn btn-primary btn-sm" 
                                                            <?php echo $index === 0 ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-arrow-up"></i>
                                                    </button>
                                                </form>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="match_id" value="<?php echo $match['id']; ?>">
                                                    <button type="submit" name="move_match_down" class="btn btn-primary btn-sm"
                                                            <?php echo $index === count($matches) - 1 ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-arrow-down"></i>
                                                    </button>
                                                </form>
                                                <a href="?edit_match=<?php echo $match['id']; ?>" class="btn btn-warning btn-sm">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="match_id" value="<?php echo $match['id']; ?>">
                                                    <button type="submit" name="delete_match" class="btn btn-danger btn-sm" 
                                                            onclick="return confirm('Are you sure you want to delete this match?')">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- Clear All Matches Form -->
                <form method="POST" id="clear-matches-form">
                    <input type="hidden" name="clear_all_matches" value="1">
                </form>
            </div>

            <!-- Categories Management -->
            <div class="section" id="categories">
                <h2 class="section-title">
                    <span>Manage Categories (<?php echo count($categories); ?>)</span>
                    <div class="section-actions">
                        <button type="button" class="btn btn-warning btn-sm" onclick="document.getElementById('clear-categories-form').submit();">
                            <i class="fas fa-trash"></i> Clear All
                        </button>
                    </div>
                </h2>
                
                <!-- Add Category Form -->
                <form method="POST" class="form-grid grid-2" style="margin-bottom: 25px;">
                    <div class="form-group">
                        <input type="text" name="category_name" required 
                               placeholder="Enter new category name (e.g., Football, Basketball)">
                    </div>
                    <div class="form-group">
                        <button type="submit" name="add_category" class="btn btn-success">
                            <i class="fas fa-plus"></i> Add Category
                        </button>
                    </div>
                </form>

                <?php if (empty($categories)): ?>
                    <div style="text-align: center; padding: 40px; color: #94a3b8;">
                        <i class="fas fa-list" style="font-size: 3rem; margin-bottom: 15px;"></i>
                        <h3>No Categories Found</h3>
                        <p>Add your first category using the form above.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Category Name</th>
                                    <th>Order</th>
                                    <th>Matches</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($categories as $index => $category): ?>
                                    <?php
                                    $category_matches = array_filter($matches, function($match) use ($category) {
                                        return $match['category'] === $category['name'];
                                    });
                                    ?>
                                    <tr>
                                        <td style="font-weight: 600;"><?php echo htmlspecialchars($category['name']); ?></td>
                                        <td><?php echo $category['order'] + 1; ?></td>
                                        <td><?php echo count($category_matches); ?></td>
                                        <td>
                                            <div class="action-buttons">
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="category_name" value="<?php echo htmlspecialchars($category['name']); ?>">
                                                    <button type="submit" name="move_category_up" class="btn btn-primary btn-sm" 
                                                            <?php echo $index === 0 ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-arrow-up"></i> Up
                                                    </button>
                                                </form>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="category_name" value="<?php echo htmlspecialchars($category['name']); ?>">
                                                    <button type="submit" name="move_category_down" class="btn btn-primary btn-sm"
                                                            <?php echo $index === count($categories) - 1 ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-arrow-down"></i> Down
                                                    </button>
                                                </form>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="category_name" value="<?php echo htmlspecialchars($category['name']); ?>">
                                                    <button type="submit" name="delete_category" class="btn btn-danger btn-sm" 
                                                            onclick="return confirm('Are you sure you want to delete this category?')">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- Clear All Categories Form -->
                <form method="POST" id="clear-categories-form">
                    <input type="hidden" name="clear_all_categories" value="1">
                </form>
            </div>
        </div>
    </div>

    <script>
        // Simple script for smooth scrolling and active section highlighting
        document.addEventListener('DOMContentLoaded', function() {
            const navLinks = document.querySelectorAll('.nav-link');
            const sections = document.querySelectorAll('.section');
            
            // Remove active class from all nav links
            navLinks.forEach(link => link.classList.remove('active'));
            
            // Add active class to current section based on hash
            if (window.location.hash) {
                const currentLink = document.querySelector(`[href="${window.location.hash}"]`);
                if (currentLink) {
                    currentLink.classList.add('active');
                }
            } else {
                // Default to dashboard
                document.querySelector('[href="#dashboard"]').classList.add('active');
            }
            
            // Handle nav link clicks
            navLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    if (this.getAttribute('href').startsWith('#')) {
                        e.preventDefault();
                        
                        // Remove active class from all nav links
                        navLinks.forEach(l => l.classList.remove('active'));
                        
                        // Add active class to clicked link
                        this.classList.add('active');
                        
                        // Scroll to section
                        const targetId = this.getAttribute('href');
                        const targetSection = document.querySelector(targetId);
                        if (targetSection) {
                            window.scrollTo({
                                top: targetSection.offsetTop - 100,
                                behavior: 'smooth'
                            });
                        }
                    }
                });
            });

            // Set default date to today
            const dateInput = document.getElementById('match_date');
            if (dateInput && !dateInput.value) {
                const today = new Date();
                const yyyy = today.getFullYear();
                const mm = String(today.getMonth() + 1).padStart(2, '0');
                const dd = String(today.getDate()).padStart(2, '0');
                dateInput.value = `${yyyy}-${mm}-${dd}`;
            }

            // Auto-update London time
            function updateLondonTime() {
                const now = new Date();
                const londonTime = now.toLocaleTimeString('en-GB', {
                    timeZone: 'Europe/London',
                    hour12: false,
                    hour: '2-digit',
                    minute: '2-digit',
                    second: '2-digit'
                });
                
                // Update all time elements
                document.querySelectorAll('.time-info').forEach(el => {
                    if (el.innerHTML.includes('London:')) {
                        el.innerHTML = `<i class="fas fa-clock"></i> London: ${londonTime}`;
                    }
                });
            }
            
            // Update time immediately and every second
            updateLondonTime();
            setInterval(updateLondonTime, 1000);
        });
    </script>
</body>
</html>