<?php
// Your PHP code remains exactly the same...
// Load matches data from JSON
$data_file = 'data/matches.json';

// Initialize matches array
$matches = [];
$categories = [];

// Check if file exists and load data
if (file_exists($data_file)) {
    $json_data = file_get_contents($data_file);
    $data = json_decode($json_data, true);
    
    if ($data && isset($data['matches']) && is_array($data['matches'])) {
        $matches = $data['matches'];
    }
    if ($data && isset($data['categories']) && is_array($data['categories'])) {
        $categories = $data['categories'];
    }
}

// Set London timezone
date_default_timezone_set('Europe/London');
$current_date = date('Y-m-d');
$current_time = date('H:i');

// Function to check if a match is live based on London time and duration
function isMatchLive($match) {
    if (!isset($match['date']) || !isset($match['time']) || !isset($match['duration'])) {
        return false;
    }
    
    $londonTz = new DateTimeZone('Europe/London');
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return false;
    }
    
    $durationParts = explode(':', $match['duration']);
    $hours = intval($durationParts[0]);
    $minutes = intval($durationParts[1] ?? 0);
    
    $endTime = clone $startTime;
    $endTime->modify("+{$hours} hours +{$minutes} minutes");
    
    $currentTime = new DateTime('now', $londonTz);
    
    return ($currentTime >= $startTime && $currentTime <= $endTime);
}

// Function to check if a match is upcoming
function isMatchUpcoming($match) {
    if (!isset($match['date']) || !isset($match['time'])) {
        return false;
    }
    
    $londonTz = new DateTimeZone('Europe/London');
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return false;
    }
    
    $currentTime = new DateTime('now', $londonTz);
    
    return $currentTime < $startTime;
}

// Function to check if stream link should be clickable (30 minutes before match)
function isStreamClickable($match) {
    if (!isset($match['date']) || !isset($match['time'])) {
        return false;
    }
    
    // Create London timezone
    $londonTz = new DateTimeZone('Europe/London');
    
    // Create match start time in London timezone
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return false;
    }
    
    // Calculate 30 minutes before start time
    $clickableTime = clone $startTime;
    $clickableTime->modify("-30 minutes");
    
    // Get current time in London
    $currentTime = new DateTime('now', $londonTz);
    
    // Check if current time is within 30 minutes before start time or after
    return $currentTime >= $clickableTime;
}

// Function to get time until stream becomes clickable
function getTimeUntilClickable($match) {
    if (!isset($match['date']) || !isset($match['time'])) {
        return 'N/A';
    }
    
    $londonTz = new DateTimeZone('Europe/London');
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return 'N/A';
    }
    
    // Calculate 30 minutes before start time
    $clickableTime = clone $startTime;
    $clickableTime->modify("-30 minutes");
    
    $currentTime = new DateTime('now', $londonTz);
    
    if ($currentTime >= $clickableTime) {
        return 'now';
    }
    
    $interval = $currentTime->diff($clickableTime);
    $hours = $interval->h;
    $minutes = $interval->i;
    
    if ($hours > 0) {
        return $hours . 'h ' . $minutes . 'm';
    } else {
        return $minutes . 'm';
    }
}

// Function to get match end time
function getMatchEndTime($match) {
    if (!isset($match['date']) || !isset($match['time']) || !isset($match['duration'])) {
        return 'N/A';
    }
    
    $londonTz = new DateTimeZone('Europe/London');
    $matchDateTime = $match['date'] . ' ' . $match['time'];
    $startTime = DateTime::createFromFormat('Y-m-d H:i', $matchDateTime, $londonTz);
    
    if ($startTime === false) {
        return 'N/A';
    }
    
    $durationParts = explode(':', $match['duration']);
    $hours = intval($durationParts[0]);
    $minutes = intval($durationParts[1] ?? 0);
    
    $endTime = clone $startTime;
    $endTime->modify("+{$hours} hours +{$minutes} minutes");
    
    return $endTime->format('H:i');
}

// Filter matches: only show matches that have stream URLs and are not expired
$upcoming_matches = array_filter($matches, function($match) {
    $has_stream = !empty(trim($match['stream_url'] ?? ''));
    
    // Check if match is live or upcoming
    $is_live = isMatchLive($match);
    $is_upcoming = isMatchUpcoming($match);
    
    return $has_stream && ($is_live || $is_upcoming);
});

// Sort matches by date and time
usort($upcoming_matches, function($a, $b) {
    $dateCompare = strcmp($a['date'] ?? '', $b['date'] ?? '');
    if ($dateCompare === 0) {
        return strcmp($a['time'] ?? '', $b['time'] ?? '');
    }
    return $dateCompare;
});

// Group matches by category
$matches_by_category = [];
foreach ($upcoming_matches as $match) {
    $category = $match['category'] ?? 'Other';
    if (!isset($matches_by_category[$category])) {
        $matches_by_category[$category] = [];
    }
    $matches_by_category[$category][] = $match;
}

// Count matches by status
$live_matches_count = count(array_filter($upcoming_matches, 'isMatchLive'));
$upcoming_matches_count = count(array_filter($upcoming_matches, 'isMatchUpcoming'));

// Count clickable streams
$clickable_streams_count = count(array_filter($upcoming_matches, 'isStreamClickable'));

// Top football teams data
$top_teams = [
    ['rank' => 1, 'name' => 'Manchester City', 'points' => 92],
    ['rank' => 2, 'name' => 'Real Madrid', 'points' => 88],
    ['rank' => 3, 'name' => 'Bayern Munich', 'points' => 85],
    ['rank' => 4, 'name' => 'Paris Saint-Germain', 'points' => 83],
    ['rank' => 5, 'name' => 'Liverpool', 'points' => 81]
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <title>Power Streams | Watch Your Favourite Sports Here</title>
    <style>
        :root {
            --primary-color: #0f172a;
            --secondary-color: #1e293b;
            --accent-color: #1E40AF;
            --accent-red: #DC2626;
            --text-color: #e2e8f0;
            --border-color: #334155;
            --success-color: #10b981;
            --table-header: #1E40AF;
            --live-color: #ef4444;
            --upcoming-color: #3b82f6;
            --disabled-color: #6b7280;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Inter', system-ui, sans-serif;
        }
        
        body {
            background-color: var(--primary-color);
            color: var(--text-color);
            line-height: 1.5;
            font-size: 15px;
        }
        
        .container {
            max-width: 1500px;
            margin: 0 auto;
            padding: 0 15px;
            display: grid;
            grid-template-columns: 1fr 320px;
            gap: 25px;
        }
        
        .main-content {
            grid-column: 1;
        }
        
        .sidebar {
            grid-column: 2;
        }
        
        /* Header */
        .top-bar {
            background: var(--secondary-color);
            padding: 12px 0;
            border-bottom: 1px solid var(--border-color);
            grid-column: 1 / -1;
        }
        
        .top-bar-content {
            max-width: 1500px;
            margin: 0 auto;
            padding: 0 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .london-time {
            background: var(--accent-color);
            padding: 6px 12px;
            border-radius: 5px;
            font-size: 13px;
            font-weight: 500;
        }
        
        .admin-link {
            background: var(--accent-red);
            color: white;
            padding: 8px 16px;
            text-decoration: none;
            border-radius: 5px;
            font-weight: 600;
            font-size: 13px;
            transition: all 0.3s ease;
        }
        
        .admin-link:hover {
            background: #b91c1c;
            transform: translateY(-2px);
        }
        
        header {
                padding: 40px 0;
                text-align: center;
                grid-column: 1 / -1;
                display: flex;
                justify-content: center;
                align-items: center;
                width: 100%;
            }
            
        .logo-container {
                width: 100%;
                display: flex;
                justify-content: center;
                align-items: center;
            }
            
        .logo {
                font-size: 3rem;
                font-weight: 800;
                color: white;
                text-align: center;
                letter-spacing: -0.5px;
            }

        .tagline {
            color: #94a3b8;
            font-size: 1.1rem;
            font-weight: 400;
            text-align: center;
        }
        
        /* Sports Navigation */
        .sports-nav {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin: -20px 0 30px 0;
            flex-wrap: wrap;
            grid-column: 1 / -1;
        }
        
        .sport-item {
            padding: 8px 16px;
            background: var(--secondary-color);
            border-radius: 18px;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 1px solid var(--border-color);
            font-weight: 500;
        }
        
        .sport-item:hover,
        .sport-item.active {
            background: var(--accent-color);
            border-color: var(--accent-color);
            transform: translateY(-2px);
        }
        
        /* Search Bar */
        .search-container {
            display: flex;
            justify-content: center;
            margin: 20px 0;
            grid-column: 1 / -1;
        }
        
        .search-bar {
            display: flex;
            width: 100%;
            max-width: 600px;
            background: var(--secondary-color);
            border-radius: 8px;
            overflow: hidden;
            border: 1px solid var(--border-color);
        }
        
        .search-input {
            flex: 1;
            padding: 12px 20px;
            background: transparent;
            border: none;
            color: var(--text-color);
            font-size: 15px;
        }
        
        .search-input::placeholder {
            color: #94a3b8;
        }
        
        .search-button {
            padding: 0 20px;
            background: var(--accent-color);
            border: none;
            color: white;
            cursor: pointer;
            font-weight: 600;
            transition: background 0.3s ease;
        }
        
        .search-button:hover {
            background: #1e3a8a;
        }
        
        /* Alert Banner */
        .alert-banner {
            background: linear-gradient(135deg, var(--accent-color), #1E3A8A);
            padding: 15px;
            text-align: center;
            border-radius: 8px;
            margin-bottom: 30px;
            
            grid-column: 1 / -1;
        }
        
        .alert-banner h2 {
            font-size: 1.3rem;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .alert-banner p {
            font-size: 14px;
            opacity: 0.9;
        }
        
        /* Quick Stats */
        .quick-stats {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
            margin-bottom: 30px;
            grid-column: 1 / -1;
        }
        
        .stat-card {
            background: var(--secondary-color);
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            border: 1px solid var(--border-color);
        }
        
        .stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            margin-bottom: 5px;
        }
        
        .stat-live {
            color: var(--live-color);
        }
        
        .stat-upcoming {
            color: var(--upcoming-color);
        }
        
        .stat-total {
            color: var(--accent-color);
        }
        
        .stat-clickable {
            color: var(--success-color);
        }
        
        .stat-label {
            font-size: 12px;
            color: #94a3b8;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        /* Matches Section */
        .matches-section {
            padding: 20px 0 40px;
        }
        
        .section-title {
            font-size: 1.5rem;
            margin-bottom: 25px;
            color: var(--text-color);
            font-weight: 600;
            text-align: center;
        }
        
        /* Category Header */
        .category-header {
            
            padding: 15px 20px;
            border-radius: 10px 10px 0 0;
            margin-bottom: 0;
            display: flex;
            align-items: center;
            justify-content: space-between;
            font-weight: 600;
            font-size: 1.3rem;
            border-bottom: 2px solid var(--accent-red);
        }
        
        .matches-count {
            background: var(--accent-red);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Table Styles */
        .matches-table {
            width: 100%;
            border-collapse: collapse;
            background: var(--secondary-color);
            border-radius: 0 0 10px 10px;
            overflow: hidden;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            font-size: 15px;
            margin-bottom: 40px;
        }
        
        .matches-table th {
            background: var(--table-header);
            padding: 16px 20px;
            text-align: left;
            font-weight: 600;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: white;
            border-bottom: 2px solid var(--accent-color);
        }
        
        .matches-table td {
            padding: 16px 20px;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }
        
        .matches-table tr:last-child td {
            border-bottom: none;
        }
        
        .matches-table tr:hover {
            background: rgba(255, 255, 255, 0.05);
        }
        
        .match-time-info {
            min-width: 140px;
        }
        
        .match-date {
            font-weight: 600;
            color: #e2e8f0;
            font-size: 14px;
        }
        
        .match-time {
            color: white;
            font-size: 13px;
            margin-top: 4px;
            font-weight: 600;
        }
        
        .match-end-time {
            color: #94a3b8;
            font-size: 12px;
            margin-top: 2px;
        }
        
        .stream-status {
            color: #94a3b8;
            font-size: 12px;
            margin-top: 4px;
            font-weight: 500;
        }
        
        .stream-available {
            color: var(--success-color);
        }
        
        .stream-countdown {
            color: var(--warning-color);
        }
        
        .league-cell {
            color: #94a3b8;
            font-size: 14px;
            font-weight: 500;
        }
        
        .teams {
            font-weight: 600;
            font-size: 15px;
        }
        
        .vs {
            color: #94a3b8;
            margin: 0 8px;
            font-size: 13px;
            font-weight: 400;
        }
        
        .action-buttons {
            display: flex;
            gap: 10px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            font-size: 13px;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            min-width: 120px;
            justify-content: center;
        }
        
        .btn-watch {
            background: var(--accent-red);
            color: white;
        }
        
        .btn-watch:hover {
            background: #b91c1c;
            box-shadow: 0 8px 16px rgba(220, 38, 38, 0.3);
        }
        
        .btn-watch.live {
            background: var(--accent-red);
            animation: pulse 2s infinite;
        }
        
        .btn-watch.disabled {
            background: var(--disabled-color);
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        .btn-watch.disabled:hover {
            background: var(--disabled-color);
            box-shadow: none;
            transform: none;
        }
        
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
        
        /* Sidebar Styles */
        .sidebar {
            display: flex;
            flex-direction: column;
            gap: 25px;
            margin-top: 20px;
        }
        
        .sidebar-widget {
            background: var(--secondary-color);
            border-radius: 10px;
            padding: 25px;
            border: 1px solid var(--border-color);
        }
        
        .widget-title {
            color: var(--text-color);
            font-size: 18px;
            font-weight: 700;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid var(--accent-color);
        }
        
        /* Teams Ranking */
        .teams-list {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        
        .team-item {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 12px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 8px;
            transition: all 0.3s ease;
            border: 1px solid transparent;
        }
        
        .team-item:hover {
            background: rgba(255, 255, 255, 0.08);
            border-color: var(--accent-color);
            transform: translateX(5px);
        }
        
        .team-rank {
            background: var(--accent-red);
            color: white;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 14px;
        }
        
        .team-name {
            flex: 1;
            font-weight: 600;
            color: var(--text-color);
        }
        
        .team-points {
            color: var(--accent-red);
            font-weight: 700;
            font-size: 14px;
        }
        
        /* Advertisement */
        .advertisement {
            background: var(--primary-color);
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            border: 2px dashed var(--border-color);
        }
        
        .ad-placeholder {
            width: 300px;
            height: 250px;
            overflow: hidden;
            border-radius: 12px;
            box-shadow: 0 0 12px rgba(0, 0, 0, 0.4);
            display: flex;
            align-items: center;
            justify-content: center;
            background: #0f111a;
            margin: 0 auto;
        }
        
        .ad-placeholder .adimage {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 12px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .ad-placeholder .adimage:hover {
            transform: scale(1.03);
            box-shadow: 0 0 20px rgba(0, 191, 255, 0.6);
        }
        
        .ad-text {
            color: #94a3b8;
            font-size: 12px;
            margin-top: 10px;
        }
        
        /* Footer */
        footer {
            background: var(--secondary-color);
            padding: 30px 0;
            border-top: 1px solid var(--border-color);
            margin-top: 40px;
            grid-column: 1 / -1;
        }
        
        .footer-content {
            text-align: center;
        }
        
        .footer-logo {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--accent-color);
            margin-bottom: 10px;
        }
        
        .copyright {
            color: #64748b;
            font-size: 13px;
            margin-top: 15px;
        }
        
        /* No Matches */
        .no-matches {
            text-align: center;
            padding: 50px 30px;
            background: var(--secondary-color);
            border-radius: 10px;
            border: 1px solid var(--border-color);
            margin: 30px 0;
        }
        
        .no-matches i {
            font-size: 3rem;
            color: #64748b;
            margin-bottom: 15px;
        }
        
        .power{
            
            color:red;
        }
        
        .no-matches h3 {
            font-size: 1.3rem;
            margin-bottom: 10px;
            color: #e2e8f0;
        }
        
        .no-matches p {
            font-size: 15px;
            color: #94a3b8;
            margin-bottom: 20px;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .container {
                grid-template-columns: 1fr;
            }
            
            .sidebar {
                grid-column: 1;
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
                gap: 20px;
            }
            
            .quick-stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        
        @media (max-width: 768px) {
            body {
                font-size: 14px;
            }
            
            .container {
                padding: 0 10px;
            }
            
            .logo {
                font-size: 1.8rem;
            }
            
            .tagline {
                font-size: 14px;
            }
            
            .matches-table {
                display: block;
                overflow-x: auto;
            }
            
            .matches-table th,
            .matches-table td {
                padding: 12px 10px;
                font-size: 13px;
            }
            
            .action-buttons {
                flex-direction: column;
                gap: 8px;
            }
            
            .btn {
                padding: 8px 12px;
                font-size: 12px;
                min-width: 100px;
            }
            
            .top-bar-content {
                flex-direction: column;
                gap: 10px;
            }
            
            .sports-nav {
                gap: 8px;
            }
            
            .sport-item {
                padding: 6px 12px;
                font-size: 13px;
            }
            
            .section-title {
                font-size: 1.3rem;
            }
            
            .category-header {
                font-size: 16px;
                padding: 12px 15px;
            }
            
            .quick-stats {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .sidebar {
                grid-template-columns: 1fr;
            }
            
            .search-bar {
                flex-direction: column;
            }
            
            .search-input {
                padding: 15px;
            }
            
            .search-button {
                padding: 12px;
                width: 100%;
            }
        }
        
        @media (max-width: 480px) {
            .matches-table {
                font-size: 13px;
            }
            
            .matches-table th,
            .matches-table td {
                padding: 10px 8px;
                font-size: 12px;
            }
            
            .teams {
                font-size: 13px;
            }
            
            .alert-banner h2 {
                font-size: 1.1rem;
            }
            
            .alert-banner p {
                font-size: 12px;
            }
            
            .ad-placeholder {
                width: 250px;
                height: 200px;
            }
            
            .category-header {
                font-size: 14px;
                padding: 10px 12px;
            }
            
            .quick-stats {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="top-bar">
        <div class="top-bar-content">
            <div class="london-time">
                <i class="fas fa-clock"></i>
                London: <span id="london-time">Loading...</span>
            </div>
            <a href="admin.php" class="admin-link">
                <i class="fas fa-lock"></i> Admin Panel
            </a>
        </div>
    </div>
    
    <header>
        <div class="logo-container">
        <div class="logo"><span class="power">Power </span> Streams</div>
        </div>
    </header>
    
    <div class="container">
        <div class="sports-nav">
            <div class="sport-item active" data-category="all">All Sports</div>
            <?php foreach ($categories as $category): ?>
                <div class="sport-item" data-category="<?php echo htmlspecialchars($category['name']); ?>">
                    <?php echo htmlspecialchars($category['name']); ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Quick Stats -->
        <div class="quick-stats">
            <div class="stat-card">
                <div class="stat-number stat-total"><?php echo count($upcoming_matches); ?></div>
                <div class="stat-label">Total Matches</div>
            </div>
            <div class="stat-card">
                <div class="stat-number stat-live"><?php echo $live_matches_count; ?></div>
                <div class="stat-label">Live Now</div>
            </div>
            <div class="stat-card">
                <div class="stat-number stat-upcoming"><?php echo $upcoming_matches_count; ?></div>
                <div class="stat-label">Upcoming</div>
            </div>
            <div class="stat-card">
                <div class="stat-number stat-clickable"><?php echo $clickable_streams_count; ?></div>
                <div class="stat-label">Streams Available</div>
            </div>
        </div>
        
        <div class="alert-banner">
            <h2>⏰ Stream Links Active 30 Minutes Before Match Start</h2>
            <p>All streams become available 30 minutes before the scheduled start time and remain active until the match ends.</p>
        </div>
        
        <!-- Main Content -->
        <main class="main-content">
            <div class="matches-section">
                <h2 class="section-title">Watch The Latest Live Sports Action — Streaming Now!</h2>
                
                <?php if (empty($upcoming_matches)): ?>
                    <div class="no-matches">
                        <i class="fas fa-tv"></i>
                        <h3>No Active Matches</h3>
                        <p>There are no live or upcoming matches with stream URLs at the moment.</p>
                        <p>Check back later for updated schedules!</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($matches_by_category as $category => $category_matches): ?>
                        <div class="category-section" data-category="<?php echo htmlspecialchars($category); ?>">
                            <!-- Category Header -->
                            <div class="category-header">
                                <span>
                                    <?php echo htmlspecialchars($category); ?> Matches
                                </span>
                                <span class="matches-count">
                                    <?php echo count($category_matches); ?> match<?php echo count($category_matches) !== 1 ? 'es' : ''; ?>
                                </span>
                            </div>
                            
                            <table class="matches-table">
                                <thead>
                                    <tr>
                                        <th>Date & Time</th>
                                        <th>League</th>
                                        <th>Match</th>
                                        <th>Stream Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($category_matches as $match): ?>
                                        <?php
                                        $is_live = isMatchLive($match);
                                        $is_clickable = isStreamClickable($match);
                                        $time_until_clickable = getTimeUntilClickable($match);
                                        $end_time = getMatchEndTime($match);
                                        $stream_url = htmlspecialchars($match['stream_url']);
                                        ?>
                                        <tr class="match-row" data-team1="<?php echo htmlspecialchars(strtolower($match['team1'])); ?>" data-team2="<?php echo htmlspecialchars(strtolower($match['team2'])); ?>">
                                            <td class="match-time-info">
                                                <div class="match-date">
                                                    <?php echo date('M j, Y', strtotime($match['date'])); ?>
                                                </div>
                                                <div class="match-time">
                                                    <?php echo $match['time']; ?> (London)
                                                </div>
                                                <?php if ($is_live): ?>
                                                    <div class="match-end-time">
                                                        Ends at: <?php echo $end_time; ?>
                                                    </div>
                                                <?php endif; ?>
                                                <div class="stream-status <?php echo $is_clickable ? 'stream-available' : 'stream-countdown'; ?>">
                                                    <?php if ($is_clickable): ?>
                                                        <i class="fas fa-check-circle"></i> Stream available
                                                    <?php else: ?>
                                                        <i class="fas fa-clock"></i> Available in <?php echo $time_until_clickable; ?>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td class="league-cell">
                                                <?php echo htmlspecialchars($match['league'] ?? 'Professional League'); ?>
                                            </td>
                                            <td>
                                                <div class="teams">
                                                    <?php echo htmlspecialchars($match['team1']); ?>
                                                    <span class="vs">vs</span>
                                                    <?php echo htmlspecialchars($match['team2']); ?>
                                                    <?php if ($is_live): ?>
                                                        <span style="color: var(--live-color); margin-left: 8px; font-size: 12px; font-weight: 600;">
                                                            ● LIVE
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if ($is_clickable): ?>
                                                    <span style="color: var(--success-color); font-weight: 600;">
                                                        <i class="fas fa-check"></i> READY
                                                    </span>
                                                <?php else: ?>
                                                    <span style="color: var(--warning-color); font-weight: 600;">
                                                        <i class="fas fa-clock"></i> COUNTDOWN
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <?php if ($is_clickable): ?>
                                                        <a href="<?php echo $stream_url; ?>" 
                                                           target="_blank" 
                                                           class="btn btn-watch <?php echo $is_live ? 'live' : ''; ?>">
                                                            <i class="fas fa-play"></i>
                                                            <?php echo $is_live ? 'WATCH LIVE' : 'WATCH LIVE'; ?>
                                                        </a>
                                                    <?php else: ?>
                                                        <span class="btn btn-watch disabled">
                                                            <i class="fas fa-clock"></i>
                                                            COMING SOON
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </main>
        
        <!-- Sidebar -->
        <aside class="sidebar">
            <!-- Top Teams Ranking -->
            <div class="sidebar-widget">
                <div class="widget-title">Top 5 Football Teams</div>
                <div class="teams-list">
                    <?php foreach ($top_teams as $team): ?>
                    <div class="team-item">
                        <div class="team-rank"><?= $team['rank'] ?></div>
                        <div class="team-name"><?= $team['name'] ?></div>
                        <div class="team-points"><?= $team['points'] ?> pts</div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Advertisement -->
            <div class="sidebar-widget">
                <div class="widget-title">Advertisement</div>
                <div class="ad-placeholder">
                    <img class="adimage" src="https://powerstreams.online/ad.png" alt="Advertisement">
                </div>
                <div class="ad-text"><script type="text/javascript">
	atOptions = {
		'key' : '2066d593945a8470994ba5b56647829c',
		'format' : 'iframe',
		'height' : 250,
		'width' : 300,
		'params' : {}
	};
</script>
<script type="text/javascript" src="//www.highperformanceformat.com/2066d593945a8470994ba5b56647829c/invoke.js"></script></div>
            </div>
        </aside>
        
    </div>
    
    
    
    <footer>
        <div class="container">
            <div class="footer-content">
                <div class="footer-logo">Power Streams</div>
                <div class="copyright">
                    &copy; <?php echo date("Y"); ?> PowerStreams. All rights reserved. | 
                    All times displayed in London Time (Europe/London) | 
                    Streams available 30 minutes before match start
                </div>
            </div>
        </div>
    </footer>

    <script>
        // Update London time - FIXED VERSION
        function updateLondonTime() {
            const londonTimeElement = document.getElementById('london-time');
            if (!londonTimeElement) return;
            
            const now = new Date();
            
            // Format time for London timezone
            const londonTime = now.toLocaleTimeString('en-GB', {
                timeZone: 'Europe/London',
                hour12: false,
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            
            londonTimeElement.textContent = londonTime;
        }
        
        // Update immediately and then every second
        updateLondonTime();
        setInterval(updateLondonTime, 1000);
        
        // Sport category filtering
        document.addEventListener('DOMContentLoaded', function() {
            const sportItems = document.querySelectorAll('.sport-item');
            const categorySections = document.querySelectorAll('.category-section');
            
            sportItems.forEach(item => {
                item.addEventListener('click', function() {
                    sportItems.forEach(i => i.classList.remove('active'));
                    this.classList.add('active');
                    
                    const selectedCategory = this.getAttribute('data-category');
                    
                    categorySections.forEach(section => {
                        if (selectedCategory === 'all' || section.getAttribute('data-category') === selectedCategory) {
                            section.style.display = 'block';
                        } else {
                            section.style.display = 'none';
                        }
                    });
                });
            });
        });
    </script>
    
    <!-- Histats.com  START  (aync)-->
    <script type="text/javascript">var _Hasync= _Hasync|| [];
    _Hasync.push(['Histats.start', '1,4853512,4,0,0,0,00010000']);
    _Hasync.push(['Histats.fasi', '1']);
    _Hasync.push(['Histats.track_hits', '']);
    (function() {
    var hs = document.createElement('script'); hs.type = 'text/javascript'; hs.async = true;
    hs.src = ('//s10.histats.com/js15_as.js');
    (document.getElementsByTagName('head')[0] || document.getElementsByTagName('body')[0]).appendChild(hs);
    })();</script>
    <noscript><a href="/" target="_blank"><img  src="//sstatic1.histats.com/0.gif?4853512&101" alt="free counter statistics" border="0"></a></noscript>
    <!-- Histats.com  END  -->
</body>
</html>